#pragma once
#include "DaisyDuino.h"
#include "luts.h"
#include "helpers.h"

enum LfoKind {
    LFO_KIND_TRI,
    LFO_KIND_SQR,
    LFO_KIND_RAND,
    LFO_KIND_JUMP,
    LFO_KIND_NONE,
};


class DaisyyLFO {
    public:
        float amount = 0.0f;
        float speed = 8.0f;
        LfoKind kind;

        DaisyyLFO();
        void Init(float sample_rate);
        float Process();
        void SetMode(LfoKind kind);
        void SetSpeed(float value);
        void SetAmount(float value);
        void PhaseAdd(float value);

    private:
        Oscillator lfo;
        SampleHold sample_and_hold;
        WhiteNoise noise;
        Metro tick;
        Easer easer;
        float value = 0.0f;
};


DaisyyLFO::DaisyyLFO() {
    this->tick.SetFreq(0.25f+this->speed*49.75f);
    this->noise.Init();
    this->lfo.SetWaveform(Oscillator::WAVE_TRI);
    this->lfo.SetAmp(1);
    this->lfo.SetFreq(this->speed);
    easer.setFactor(0.001);
}

void DaisyyLFO::Init(float sample_rate) {
    this->tick.Init(10, sample_rate);
    this->lfo.Init(sample_rate);
}

float DaisyyLFO::Process() {
    this->lfo.SetFreq(this->speed);
    this->tick.SetFreq(0.15f+this->speed);

    switch (this->kind) {
        case LfoKind::LFO_KIND_TRI:
            this->lfo.SetWaveform(Oscillator::WAVE_TRI);
            this->value = this->lfo.Process() * 0.2f;
            break;
        case LfoKind::LFO_KIND_SQR:
            this->lfo.SetWaveform(Oscillator::WAVE_SQUARE);
            this->value = this->lfo.Process() * 0.2f;
            break;
        case LfoKind::LFO_KIND_RAND:
        case LfoKind::LFO_KIND_JUMP:
            // Get a clock and some noise
            uint8_t trig = tick.Process();
            float noise_value = noise.Process();
            // Set the easer
            if (this->kind == LfoKind::LFO_KIND_RAND) {
                this->easer.setFactor(0.01);
            } else {
                // Easer is basically deactivated
                this->easer.setFactor(1.0);
            }
            // Get a random number from the sample and hold
            float rand = this->easer.Process(
                this->sample_and_hold.Process(
                    trig, 
                    noise_value * 5.0f, 
                    this->sample_and_hold.MODE_SAMPLE_HOLD
            ));

            // If there is a trigger set a new random value
            if (trig) {
                switch (this->kind) {
                    case LfoKind::LFO_KIND_RAND:
                        this->value = rand;
                        break;
                    case LfoKind::LFO_KIND_JUMP:
                        // Trigger more often if the amount is higher
                        if (drand(0.0f, 1.0f) < this->amount) {
                            this->value = rand * 16000.0f;
                            return this->value * this->amount;
                        }
                        break;
                }
            }
            break;
    }

    // The Jump LFO only returns non-zero values when it is actually triggered (see above)
    if (this->kind == LFO_KIND_JUMP) {
        return 0.0f;
    }

    return this->value * this->amount;
}

void DaisyyLFO::SetMode(LfoKind kind) {
    this->kind = kind;
}

void DaisyyLFO::SetSpeed(float value) {
    this->speed = value;
}

void DaisyyLFO::SetAmount(float value) {
    this->amount = sqrt(max(0.01f, value) - 0.01f);
}

void DaisyyLFO::PhaseAdd(float value) {
    this->lfo.PhaseAdd(value);
}

