#include <stdint.h>
#include "WSerial.h"
#include "usbd_def.h"
#ifndef Ui_h
#define Ui_h

#ifdef DEBUG
#define DEBUG_PRINT(x) Serial.print (x)
#define DEBUG_PRINTLN(x) Serial.println (x)
#else
#define DEBUG_PRINT(x)
#define DEBUG_PRINTLN(x)
#endif


#include "Adafruit_SH110X.h"
#include "Adafruit_GFX.h"
#include "potentiometers.h"
#include "buttons.h"
#include "looper.h"
#include "button_grid.h"

#define UI_MAX_FPS 60
#define WAVEFORM_OVERSAMPLING 2
#define WAVEFORM_LIN true

extern Potentiometer pot_1, pot_2, pot_3, pot_4, pot_5, pot_6, pot_7;
extern Button button_1, button_2, button_3, button_4, button_5, button_6;
extern Adafruit_SH1106G display;
extern atoav::Looper looper;

// Should the splash-screen be shown on boot?
bool show_splash = true;

// Represents the possible states of the UI
enum UiMode {
  UI_MODE_SPLASH,         // A splash screen that is shown on startup
  UI_MODE_DEFAULT,        // Default screen: Show Waveform and Parameters
  UI_MODE_REC_MENU,       // ButtonGrid Menu: Recording Settings
  UI_MODE_PLAY_MENU,      // ButtonGrid Menu: Playback Settings
  UI_MODE_TRIGGER_MENU,   // ButtonGrid Menu: Trigger Settings
  UI_MODE_FX_MENU,        // ButtonGrid Menu: FX Settings
  UI_MODE_LAST
};

// Represents possible recording modes
enum RecMode {
  REC_MODE_FULL,          // Record into full buffer (looping back to start)
  REC_MODE_LOOP,          // Limit recording to the loop
  REC_MODE_FULL_SHOT,     // Record into full buffer, stop at the end
  REC_MODE_LAST
};

// Represents possible recording sources
enum RecSource {
  REC_SOURCE_PRE,            // Record Incoming audio
  REC_SOURCE_POST,           // Record effects
  REC_SOURCE_OUT,            // Record the buffer output
  REC_SOURCE_NOISE,          // Record Noise
  REC_SOURCE_LAST
};

// Represents possible playback modes
enum PlayMode {
  PLAY_MODE_DRUNK,        // Drunken Walk
  PLAY_MODE_WINDOW,       // Sliding window
  PLAY_MODE_LOOP,         // Loop
  PLAY_MODE_GRAIN,        // Granular ?
  PLAY_MODE_ONESHOT,      // Play it once
  PLAY_MODE_LAST
};

// Represents possible recording states
enum RecordingState {
  REC_STATE_NOT_RECORDING, // Not recording
  REC_STATE_RECORDING,     // Recording (replace what is in the buffer)
  REC_STATE_OVERDUBBING,   // Overdubbing (mix recorded values with the existing samples)
  REC_STATE_LAST
};

// The Ui is _the_ coordinating class for the whole interaction.
// The default mode
// Note Descriptions get a space of 21 chars and 8 lines
class Ui {
  public:
    Ui() : button_grids {
      ButtonGrid((int) UI_MODE_REC_MENU, {
        GridButton("REC\nMENU", &button_1, true),
        GridButton("MOM\nTOGGLE", &button_2, false, BUTTON_TYPE_TOGGLE, 0, "REC/OVERDUB BUTTONS\n\nMOM->Record while    \n     Button is held\n\nTOGGLE->Press once to\n        start, press \n        again to stop"),
        GridButton("PRE\nPOST\nOUT\nNOISE", &button_3, false, BUTTON_TYPE_MULTITOGGLE, 0, "REC/OVERDUB SOURCE\n\nPRE----->Direct Input\nPOST---->With Effects\nOUT---->Looper Output\nNOISE--->Noise Source"),
        GridButton("FULL\nLOOP\nSHOT", &button_4, false, BUTTON_TYPE_MULTITOGGLE, 1, "RECORDING REGION\n\nFULL---->Whole Buffer\nLOOP----->Loop Bounds\nSHOT->Full Buffer but\n      stop at the end"),
        GridButton("NORMAL\nPITCHD\nUNPTCH", &button_5, false, BUTTON_TYPE_MULTITOGGLE, 0, "SPEED OF THE REC HEAD\n\nNORMAL--->Fixed Speed\nPITCHD->Inverse Playh\nUNPITCH----->Playhead\n             Speed"),
        GridButton("START\nLOOPST\nPLAYHD", &button_6, false, BUTTON_TYPE_MULTITOGGLE, 0, "START RECORDING AT\n\nSTART--->Start of the\n         Buffer\nLOOP---->Start of the\n         Loop\nPLAYH--->Position of\n         the Playhead"),
      }),
      ButtonGrid((int) UI_MODE_PLAY_MENU, {
        GridButton("LOOP", &button_1, false),
        GridButton("PLAY\nMENU", &button_2, true),
        GridButton("WINDOW", &button_3, false),
        GridButton("DRUNK", &button_4, false),
        GridButton("GRAIN", &button_5, false),
        GridButton("SHOT", &button_6, false),
      }),
      ButtonGrid((int) UI_MODE_TRIGGER_MENU, {
        GridButton("MIDI\nTRIG.", &button_1, false),
        GridButton("MIDI\nUNMUTE", &button_2, false),
        GridButton("TRIG.\nMENU", &button_3, true),
        GridButton("MANUAL\nTRIG.", &button_4, false),
        GridButton("MANUAL\nUNMUTE", &button_5, false),
        GridButton("AUTO", &button_6, false),
      }),
      ButtonGrid((int) UI_MODE_FX_MENU, {
        GridButton("DELAY", &button_1, false),
        GridButton("REVERB", &button_2, false),
        GridButton("-", &button_3, false),
        GridButton("-", &button_4, false),
        GridButton("-", &button_5, false),
        GridButton("FX\nMENU", &button_6, true),
      }),
    } {};

    // Store the Button Grids declared above (make sure the lenght matches!)
    ButtonGrid button_grids[4];

    // Stores the current Ui Mode
    UiMode ui_mode = UI_MODE_SPLASH;

    // Default Recording Mode
    RecMode rec_mode = REC_MODE_LOOP;

    // Default Recording Source
    RecSource rec_source = REC_SOURCE_PRE;

    // Set the mode of the UI (and thus change the screen)
    void setMode(UiMode mode) {
      if (last_ui_mode == mode) { return; }
      last_ui_mode = ui_mode;
      ui_mode = mode;
      switch (ui_mode) {
        case UI_MODE_SPLASH:
          break;
        case UI_MODE_DEFAULT:
          break;
        case UI_MODE_REC_MENU:
          break;
        case UI_MODE_PLAY_MENU:
          break;
        case UI_MODE_TRIGGER_MENU:
          break;
        case UI_MODE_FX_MENU:
          break;
      }
    }

    // Render the UI
    // Except for the splash screen this is expected to be called
    // repeatedly in a loop
    void Render() {
      switch (ui_mode) {
        case UI_MODE_SPLASH:
          renderSplash();
          break;
        case UI_MODE_DEFAULT:
          setupDefault();
          renderDefault();
          break;
        case UI_MODE_REC_MENU:
          setupRecMenu();
          renderGrid(0, rec_mode);
          break;
        case UI_MODE_PLAY_MENU:
          renderGrid(1);
          break;
        case UI_MODE_TRIGGER_MENU:
          renderGrid(2);
          break;
        case UI_MODE_FX_MENU:
          renderGrid(3);
          break;
      }
    }

    // Helper method to render a certain button grid
    void renderGrid(size_t num, int button_enum=0) {
      double now = millis();
      if (now - last_render > (1000.0/UI_MAX_FPS)) {
        display.clearDisplay();
        button_grids[num].render(button_enum);
        // Display all that stuff and store the time of the last render
        display.display();
        last_render = now;
      }
    }

    // Renders a splash screen (runs once)
    void renderSplash() {
      display.setTextSize(1);
      if (show_splash) {
        display.setTextColor(SH110X_BLACK);
        // Play a fancy intro splash screen
        for (int i=0; i < 91; i++) {
          display.clearDisplay();
          display.fillCircle(display.width()/2, display.height()/2, i, SH110X_WHITE);
          display.fillCircle(display.width()/2, display.height()/2, max(0, i-2), SH110X_BLACK);
          if (i < 10) {
            centeredText("I", display.width()/2, display.height()/2-4, SH110X_WHITE);
            centeredText("O", display.width()/2, display.height()/2+4, SH110X_WHITE);
          } else if (i < 20) {
            centeredText("I S", display.width()/2, display.height()/2-4, SH110X_WHITE);
            centeredText("O P", display.width()/2, display.height()/2+4, SH110X_WHITE);
          } else if (i < 40) {
            centeredText("A I S", display.width()/2, display.height()/2-4, SH110X_WHITE);
            centeredText("O O P", display.width()/2, display.height()/2+4, SH110X_WHITE);
          } else if (i < 50) {
            centeredText("A I S Y", display.width()/2, display.height()/2-4, SH110X_WHITE);
            centeredText("O O P E", display.width()/2, display.height()/2+4, SH110X_WHITE);
          } else if (i < 60) {
            centeredText("D A I S Y", display.width()/2, display.height()/2-4, SH110X_WHITE);
            centeredText("L O O P E", display.width()/2, display.height()/2+4, SH110X_WHITE);
          } else {
            centeredText("D A I S Y Y", display.width()/2, display.height()/2-4, SH110X_WHITE);
            centeredText("L O O P E R", display.width()/2, display.height()/2+4, SH110X_WHITE);
          }
          
          display.display();
          delay(1);
        }
        display.invertDisplay(true);
        display.display();
        delay(800);
        display.clearDisplay();
        display.invertDisplay(false);
        display.display();
      }

      // Splash rendering is now done, go to next UI Mode
      setMode(UI_MODE_DEFAULT);
    }

    // Helper method to reset the controls
    void resetControls() {
      button_1.reset();
      button_2.reset();
      button_3.reset();
      button_4.reset();
      button_5.reset();
      button_6.reset();
    }

    Button* setupButtonGrid(int n) {
      Serial.println("[UI] Setup Rec Menu");

      // Find the index of the home button
      int home_button_index = button_grids[n].homeButtonIndex();

      // Create a pointer to the hoime button
      Button* home_button = button_grids[n].grid_buttons_[home_button_index].button;

      // Reset the controls
      resetControls();

      // Setup the button grid
      button_grids[n].setup();

      // Return to default mode on release
      home_button->onReleased([this, n](){
        this->setMode(UI_MODE_DEFAULT);
        this->button_grids[n].hideAllDescriptions();
      });

      // Return pointer to the home button
      return home_button;
    }

    // Setup the Recording Menu
    void setupRecMenu() {
      // Only run once when the ui_mode changed
      if (ui_mode == UI_MODE_REC_MENU && last_ui_mode != UI_MODE_REC_MENU) {
        int n = 0;

        // Setup button Grid
        Button* home_button = setupButtonGrid(n);

        // Toggle between momentary and toggle recording modes
        button_2.onPress([this, n](){       
          Serial.print("[UI] Mom/Toggle option ");
          if (rec_button_momentary) {
            Serial.println("momentary -> toggle");
          }else{
            Serial.println("toggle -> momentary");
          }
          rec_button_momentary = !rec_button_momentary;
          button_grids[n].grid_buttons_[1].active = !rec_button_momentary;
        });

        
        // Set Recording Source (Pre/Post/Out/Noise)
        button_3.onPress([this, n](){
          button_grids[n].grid_buttons_[2].next();
          rec_source = (RecSource) button_grids[n].grid_buttons_[2].active;
        }); 

        // Switch Recording modes (Full/Loop/Oneshot)
        button_4.onPress([this, n](){ 
          button_grids[n].grid_buttons_[3].next();
          // Button.active returns number according to mode, we cast it to a RecMode enum
          rec_mode = (RecMode) button_grids[n].grid_buttons_[3].active;
          switch (rec_mode) {
            case REC_MODE_FULL: looper.setRecModeFull(); break;
            case REC_MODE_LOOP: looper.setRecModeLoop(); break;
            case REC_MODE_FULL_SHOT: looper.setRecModeFullShot(); break;
          }
        });

        // Set Recording Pitch mode (Normal/Pitched/Unpitched)
        button_5.onPress([this, n](){ 
          button_grids[n].grid_buttons_[4].next();
          looper.setRecPitchMode((atoav::RecPitchMode) button_grids[n].grid_buttons_[4].active);
        });

        // Set Recording Start Option (Buffer Start/Loop Start/Playhead)
        button_6.onPress([this, n](){
          button_grids[n].grid_buttons_[5].next();
          looper.setRecStartMode((atoav::RecStartMode) button_grids[n].grid_buttons_[5].active);
        });

        // Store the last ui mode, for the check on top
        last_ui_mode = ui_mode;
      }
    }

    // Setup the default (waveform) screen
    void setupDefault() {
      // Only run once on mode change
      if (ui_mode == UI_MODE_DEFAULT && last_ui_mode != UI_MODE_DEFAULT) {
        Serial.println("[UI] Setup Default mode");

        // Reset controls
        resetControls();

        // Set up the initial recording mode
        switch (rec_mode) {
          case REC_MODE_FULL: looper.setRecModeFull(); break;
          case REC_MODE_LOOP: looper.setRecModeLoop(); break;;
          case REC_MODE_FULL_SHOT: looper.setRecModeFullShot(); break;
        };

        // Setup Button functions (these should enter the ButtonGrid Menus)
        button_1.onHold([this](){ this->setMode(UI_MODE_REC_MENU); });
        button_2.onHold([this](){ this->setMode(UI_MODE_PLAY_MENU); });
        button_3.onHold([this](){ this->setMode(UI_MODE_TRIGGER_MENU); });
        button_6.onHold([this](){ this->setMode(UI_MODE_FX_MENU); });

        // Set the recording/overdub buttons to toggle or momentary
        // depending on the value of the option
        if (rec_button_momentary) {
          Serial.println("[UI] Set to momentary mode");
          button_4.onHold([this](){ this->activateRecording(); });
          button_5.onHold([this](){ this->activateOverdub(); });
          button_4.onReleased([this](){ this->stopRecording(); });
          button_5.onReleased([this](){ this->stopRecording(); });
        } else {
          Serial.println("[UI] Set to toggle mode");
          button_4.onReleased([this](){ this->toggleRecording(); });
          button_5.onReleased([this](){ this->toggleOverdub(); });
        }
        
        // Store the last ui mode, for the check on top
        last_ui_mode = ui_mode;
      }
    }

    // Render the default screen (waveform)
    void renderDefault() {
      // Store the current time and check how long ago the last frame was
      // in ms
      double now = millis();
      if (now - last_render > (1000.0/UI_MAX_FPS)) {

        // Clear the display
        display.clearDisplay();

        // Waveform should be maximum screen-heigh
        int wave_height = display.height() * 1.0f;
        // Ensure that when stepping from left to right we fit the waveform on the screen
        int step = looper.getBufferLength() / (display.width() * WAVEFORM_OVERSAMPLING);
        // Helper variable for the bottom of the screen
        int bottom = display.height()-1;

        // Render the waveform by iterating through the samples (oversampled by a factor
        // defined on top of this file). Average the samples for each pixel of the 128 px
        // wide screen and cache the resulting heights so we only have to recalculate when
        // the waveform changes
        for (int i=0; i<display.width()*WAVEFORM_OVERSAMPLING; i+=WAVEFORM_OVERSAMPLING) {
          uint16_t x = int(i / WAVEFORM_OVERSAMPLING);
          // Only recalculate if the cahce is dirty, else use cache
          if (waveform_cache_dirty) {
            float sig = 0.0f;
            float scale = 1.0f;
            if (!WAVEFORM_LIN) {
              scale = 10.0f;
            }
            // Step through the buffer and sum the absolute values
            for (int s=0; s<WAVEFORM_OVERSAMPLING; s++) {
              float abs_sig = looper.getBuffer()[step*i];
              abs_sig = abs(abs_sig) * scale;
              sig += abs_sig;
            }
            // We oversampled so divide here
            sig = sig / float(WAVEFORM_OVERSAMPLING);

            if (!WAVEFORM_LIN) {
              // Volume is logarithmic (hiding silent noises)
              if (sig != 0.0f) {
                sig = log10(sig);
              }
            }
            waveform_cache[x] = int(sig * wave_height);
          }

          // Draw the vertical lines from bottom up, depending on the level of the
          // calulcated wave on this point of the screen
          display.drawFastVLine(x, bottom, -waveform_cache[x], SH110X_WHITE);
          
        }
        // Draw one horizontal line on bottom
        display.drawFastHLine(0, bottom, display.width(), SH110X_WHITE);

        // Cache is now marked as clean
        waveform_cache_dirty = false;

        // Draw Indicator for loop start 
        int x_start_loop = int(loop_start * display.width());
        display.drawLine(x_start_loop, 0, x_start_loop, bottom, SH110X_WHITE);
        display.fillTriangle(x_start_loop, 6, x_start_loop, 0, x_start_loop+3, 0, SH110X_WHITE);

        // Draw Indicator for Loop End
        int x_loop_length = int(loop_length * display.width());
        int x_loop_end = (x_start_loop + x_loop_length) % display.width();
        display.drawLine(x_loop_end, 0, x_loop_end, bottom, SH110X_WHITE);
        display.fillTriangle(x_loop_end, 6, x_loop_end-3, 0, x_loop_end, 0, SH110X_WHITE);

        // Draw connecting line for start and end
        if (x_loop_end >= x_start_loop) {
          display.drawLine(x_start_loop, 0, x_loop_end, 0, SH110X_WHITE);
        } else {
          display.drawLine(x_start_loop, 0, display.width(), 0, SH110X_WHITE);
          display.drawLine(0, 0, x_loop_end, 0, SH110X_WHITE);
        }

        // Draw Playhead
        int x_playhead = int(looper.GetPlayhead() * display.width()) + x_start_loop;
        display.drawLine(x_playhead, 6, x_playhead, 24, SH110X_WHITE);

        // Draw Recording Indicator and Recording Head
        if (recording_state == REC_STATE_RECORDING) {
          // Draw Rec Head
          int x_rec_head = int(looper.GetRecHead() * display.width());
          display.drawLine(x_rec_head, 10, x_rec_head, bottom, SH110X_WHITE);
          display.fillCircle(x_rec_head, 10, 3, SH110X_WHITE);
          // Record sign
          display.fillRect(0, 0, 13, 13, SH110X_WHITE);
          display.fillRect(2, 2, 12, 12, SH110X_WHITE);
          display.fillRect(1, 1, 11, 11, SH110X_BLACK);
          display.fillCircle(6, 6, 3, SH110X_WHITE);
        }

        // Draw Overdub Indicator and Recording Head
        if (recording_state == REC_STATE_OVERDUBBING) {
          // Draw Rec Head
          int x_rec_head = int(looper.GetRecHead() * display.width());
          display.drawLine(x_rec_head, 10, x_rec_head, bottom, SH110X_WHITE);
          display.fillCircle(x_rec_head, 10, 3, SH110X_WHITE);

          // Overdub sign (a "plus")
          display.fillRect(0, 0, 13, 13, SH110X_WHITE);
          display.fillRect(2, 2, 12, 12, SH110X_WHITE);
          display.fillRect(1, 1, 11, 11, SH110X_BLACK);
          display.drawLine(6, 2, 6, 10, SH110X_WHITE);
          display.drawLine(2, 6, 10, 6, SH110X_WHITE);
        }

        // Render potentiometer UIs in case a knob is changed
        pot_1.renderUi();
        pot_2.renderUi();
        pot_3.renderUi();
        pot_4.renderUi();
        pot_5.renderUi();
        pot_6.renderUi();
        pot_7.renderUi();

        // Display all the things done above
        display.display();

        // Store the time of when we started rendering
        last_render = now;
      }
    }

    // Activate recording and set the waveform cache to dirty
    void activateRecording() {
      if (recording_state != REC_STATE_RECORDING) {
        Serial.println("[UI] Activate Recording");
        looper.SetRecording(true, false);
        waveform_cache_dirty = true;
        recording_state = REC_STATE_RECORDING;
      }
    }

    // Toggle recording
    void toggleRecording() {
      Serial.print("[UI] Toggle Recording ");
      Serial.print(recording_state);
      switch (recording_state) {
        case REC_STATE_NOT_RECORDING:
          activateRecording();
          break;
        case REC_STATE_RECORDING:
          stopRecording();
          break;
        case REC_STATE_OVERDUBBING:
          activateRecording();
          break;
      }
      Serial.print(" -> ");
      Serial.println(recording_state);
    }

    // Returns true if we are recording
    bool isRecording() {
      return looper.isRecording();
    }

    // Activates overdubbing
    void activateOverdub() {
      if (recording_state != REC_STATE_OVERDUBBING) {
        Serial.println("[UI] Activate Overdub");
        waveform_cache_dirty = true;
        recording_state = REC_STATE_OVERDUBBING;
        looper.SetRecording(true, true);
      }
    }

    // Stop the recording
    void stopRecording() {
      if (recording_state != REC_STATE_NOT_RECORDING) {
        Serial.println("[UI] Stop Recording");
        recording_state = REC_STATE_NOT_RECORDING;
        looper.SetRecording(false, false);
      }
    }

    // Toggle overdub off and on
    void toggleOverdub() {
      Serial.print("[UI] Toggle Overdub ");
      Serial.print(recording_state);
      switch (recording_state) {
        case REC_STATE_NOT_RECORDING:
          activateOverdub();
          break;
        case REC_STATE_OVERDUBBING:
          stopRecording();
          break;
        case REC_STATE_RECORDING:
          activateOverdub();
          break;
      }
      Serial.print(" -> ");
      Serial.println(recording_state);
    }

    // Return true if overdub is running
    bool isOverdubbing() {
      return looper.isOverdubbing();
    }

    // Reset the recording state (mark waveform cahce dirty)
    void resetRecordingState() {
      if (recording_state == REC_STATE_RECORDING || recording_state == REC_STATE_OVERDUBBING) {
        waveform_cache_dirty = true; 
      }
    }

    // Update the Ui variables (expected to run repeatedly)
    void update() {
      resetRecordingState();
    }

    // Set the Looper start/length to a given value
    void setLoop(float start, float length) {
      loop_start = start;
      loop_length = length;
      looper.SetLoop(start, length);
    }

    private:
      double last_render;
      uint16_t waveform_cache[128] = {0};
      bool waveform_cache_dirty = true;
      RecordingState recording_state = REC_STATE_NOT_RECORDING;
      UiMode last_ui_mode = UI_MODE_LAST;
      bool rec_button_momentary = true;
      float loop_start = 0.0f;
      float loop_length = 0.0f;
};



#endif