#ifndef Looper_h
#define Looper_h

namespace atoav {

enum RecPitchMode {
  REC_PITCH_MODE_NORMAL,
  REC_PITCH_MODE_PITCHED,
  REC_PITCH_MODE_UNPITCHED,
  REC_PITCH_MODE_LAST,
};

enum RecStartMode {
  REC_START_MODE_BUFFER,
  REC_START_MODE_LOOP,
  REC_START_MODE_PLAYHEAD,
  REC_START_MODE_LAST,
};

class Looper {
  public:
    void Init(float *buf, size_t length) {
      buffer = buf;
      buffer_length = length;
      // Reset buffer contents to zero
      memset(buffer, 0, sizeof(float) * buffer_length);
    }

    RecPitchMode rec_pitch_mode = REC_PITCH_MODE_NORMAL;
    RecStartMode rec_start_mode = REC_START_MODE_BUFFER;

    void SetRecording(bool is_recording, bool is_overdub) {
      this->is_overdub = is_overdub;
      this->is_recording = is_recording || is_overdub;
      //Initialize recording head position on start
      // if (rec_env_pos_inc <= 0 && is_recording) {
      if (is_recording) {
        // rec_head = (loop_start + play_head) % buffer_length;
        switch (rec_start_mode) {
          case REC_START_MODE_LOOP:
            rec_head = (loop_start) % buffer_length;
            break;
          case REC_START_MODE_BUFFER:
            rec_head = 0.0f;
            break;
          case REC_START_MODE_PLAYHEAD:
            rec_head = fmod(loop_start + play_head, float(buffer_length));
            break;
        }
          
        is_empty = false;
      }
      // When record switch changes state it effectively
      // sets ramp to rising/falling, providing a
      // fade in/out in the beginning and at the end of 
      // the recorded region.
      rec_env_pos_inc = is_recording ? 1 : -1;
    }

    void SetLoop(const float loop_start_time, const float loop_length_time) {
      // Set the start of the next loop
      pending_loop_start = static_cast<size_t>(loop_start_time * (buffer_length - 1));

      // If the current loop start is not set yet, set it too
      if (!is_loop_set) loop_start = pending_loop_start;

      // Set the length of the next loop
      pendingloop_length = max(kMinLoopLength, static_cast<size_t>(loop_length_time * buffer_length));

      // CHECK if this is truly good
      // loop_length = pendingloop_length;
      // loop_length = pendingloop_length;

      //If the current loop length is not set yet, set it too
      if (!is_loop_set) loop_length = pendingloop_length;
      is_loop_set = true;
    }
  
    void Record(float in) {
      // Calculate iterator position on the record level ramp.
      if (rec_env_pos_inc > 0 && rec_env_pos < kFadeLength
       || rec_env_pos_inc < 0 && rec_env_pos > 0) {
          rec_env_pos += rec_env_pos_inc;
      }
      // If we're in the middle of the ramp - record to the buffer.
      if (rec_env_pos > 0) {
        // Calculate fade in/out
        float rec_attenuation = static_cast<float>(rec_env_pos) / static_cast<float>(kFadeLength);
        if (this->is_overdub) {
          buffer[int(rec_head)] += in * rec_attenuation;
        } else {
          buffer[int(rec_head)] = in * rec_attenuation + buffer[int(rec_head)] * (1.f - rec_attenuation);
        }

        // Set recording pitch mode
        switch (rec_pitch_mode) {
          case REC_PITCH_MODE_NORMAL: 
            rec_head += 1.0f; 
            break;
          case REC_PITCH_MODE_UNPITCHED: 
            rec_head += playback_increment; 
            break;
          case REC_PITCH_MODE_PITCHED:
            if (playback_increment != 0.0) {
              rec_head += 1.0f/playback_increment;
            }
            break;
        }

        // Different recording modes
        if (!stop_after_recording) {
          if (!stay_within_loop) {
            // record into whole buffer
            rec_head = fmod(rec_head, float(buffer_length));
          } else {
            // Limit rec head to stay inside the loop
            rec_head = fmod(rec_head, float(loop_start + loop_length));
            rec_head = max(float(loop_start), rec_head);
          }
        } else {
          if (!stay_within_loop) {
            if (rec_head > buffer_length) { SetRecording(false, false); }
          } else {
            if (rec_head > loop_start + loop_length) { SetRecording(false, false); }
          }
        }

        if (rec_head > buffer_length) {
          rec_head = 0.0f;
        } else if (rec_head < 0) {
          rec_head = buffer_length;
        }
        
      }
    }

    float Process() {
      // Early return if the buffer is empty
      if (is_empty) {
        return 0;
      }

      // Variables for the Playback from the Buffer
      float attenuation = 1;
      float output = 0;

      // Calculate fade in/out
      if (play_head < kFadeLength) {
        attenuation = static_cast<float>(play_head) / static_cast<float>(kFadeLength);
      }
      else if (play_head >= loop_length - kFadeLength) {
        attenuation = static_cast<float>(loop_length - play_head) / static_cast<float>(kFadeLength);
      }
      
      // Ensure we are actually inside the buffer
      auto play_pos = int(loop_start + play_head) % buffer_length;

      // Read from the buffer
      output = buffer[play_pos] * attenuation;

      // Advance playhead by the increment
      play_head += playback_increment;

      // Ensure the playhead stays within bounds of the loop
      if (play_head >= loop_length) {
        loop_start = pending_loop_start;
        loop_length = pendingloop_length;
        play_head = 0;
      } else if (play_head <= 0) {
        loop_start = pending_loop_start;
        loop_length = pendingloop_length;
        play_head = loop_length;
      }
      
      // Return the attenuated signal
      return output * attenuation;
    }

    float GetPlayhead() {
      return  float(play_head) / float(buffer_length);
    }

    float GetRecHead() {
      return  float(rec_head) / float(buffer_length);
    }

    bool toggleRecMode() {
      stay_within_loop = !stay_within_loop;
      return stay_within_loop;
    }

    void setRecModeFull() {
      Serial.println("[Looper] Set RecMode to Full");
      stay_within_loop = false;
      stop_after_recording = false;
    }

    void setRecModeLoop() {
      Serial.println("[Looper] Set RecMode to Loop");
      stay_within_loop = true;
      stop_after_recording = false;
    }

    void setRecModeFullShot() {
      Serial.println("[Looper] Set RecMode to Loop FullShot");
      stay_within_loop = false;
      stop_after_recording = true;
    }

    void setRecModeLoopShot() {
      Serial.println("[Looper] Set RecMode to Loop Oneshot");
      stay_within_loop = true;
      stop_after_recording = true;
    }


    void setPlaybackSpeed(float increment) {
      playback_increment = increment;
    }

    void addToPlayhead(float value) {
      play_head += value;
    }

    float* getBuffer() {
      return buffer;
    }

    size_t getBufferLength() {
      return buffer_length;
    }

    bool isRecording() {
      return is_recording;
    }

    bool isOverdubbing() {
      return is_overdub;
    }

    void setRecPitchMode(RecPitchMode mode) {
      Serial.print("[Looper] Set RecPitchMode from ");
      Serial.print(rec_pitch_mode);
      Serial.print(" to ");
      Serial.println(mode);
      rec_pitch_mode = mode;
    }

    void setRecStartMode(RecStartMode mode) {
      Serial.print("[Looper] Set RecStartMode from ");
      Serial.print(rec_start_mode);
      Serial.print(" to ");
      Serial.println(mode);
      rec_start_mode = mode;
    }

  private:
    static const size_t kFadeLength = 200;
    static const size_t kMinLoopLength = 2 * kFadeLength;

    float* buffer;

    size_t buffer_length       = 0;
    size_t loop_length         = 0;
    size_t pendingloop_length = 0;
    size_t loop_start          = 0;
    size_t pending_loop_start  = 0;

    float play_head = 0.0f;
    float rec_head  = 0.0f;

    float playback_increment = 1.0f;

    size_t rec_env_pos      = 0;
    int32_t rec_env_pos_inc = 0;
    bool is_empty  = true;
    bool is_loop_set = false;
    bool stay_within_loop = false;
    bool is_overdub = false;
    bool is_recording = false;
    bool stop_after_recording = false;
};

}; // namespace atoav

#endif