#!/usr/bin/python3

import sys
import math
import json
import time
import cv2
import numpy as np
import os
import jack


# load the configuration
conf = json.load(open('conf.json'))

# use RasPiCam capture device, or dummy file if filename is given
if len(sys.argv)>1:
  img = cv2.imread(sys.argv[1])
  img = cv2.resize(img,(640, 480))
  rawCapture=None
else:
  # use Raspi Camera
  from picamera.array import PiRGBArray
  from picamera import PiCamera
  # initialize the camera and grab a reference to the raw camera capture
  camera = PiCamera()
  camera.resolution = tuple(conf["resolution"])
  camera.framerate = conf["fps"]
  camera.exposure_mode='off'
  camera.shutter_speed=conf["shutter_milliseconds"]*1000
  rawCapture = PiRGBArray(camera, size=tuple(conf["resolution"]))
  # allow the camera to warmup
  print("[INFO] warming up...")
  time.sleep(conf["camera_warmup_time"])


# we are a Jack MIDI jack_client to send MIDI messages
jack_client = jack.Client("Woodsound Scanner")
midi_outport = jack_client.midi_outports.register("output")
midi_output_queue=[] # MIDI messages ready to send

# Jack MIDI jack_client output callback - feeds MIDI messages to Jack on it's demand
@jack_client.set_process_callback
def process(frames):
  midi_outport.clear_buffer()
  while midi_output_queue:
    midi_outport.write_midi_event(0,midi_output_queue.pop())
jack_client.activate()


histogram_steps=conf["bins"]
threshold=conf["threshold"]

# capture frames from the camera
# clahe=cv2.createCLAHE(clipLimit=conf["contrast_limit"])
while True:

    if not rawCapture is None:
      # clear the stream in preparation for the next frame
      rawCapture.truncate(0)
      # capture frame
      camera.capture(rawCapture, format="bgr", use_video_port=True)
      frame = rawCapture.array
    else:
      frame = img
    
    # crop down to some rectangular bar used for processing
    # cropped_frame=frame[100:280,0:640]
    cropped_frame=frame[conf['y_min']:conf['y_max'],5:-5]


    # histogram output
    # crop single line near bottom for color histogram output
    line_frame=cropped_frame[-2:-1,:]
    # compute b/w histogram
    line_frame = cv2.cvtColor(line_frame, cv2.COLOR_BGR2GRAY)
    histogram=np.zeros(histogram_steps,dtype=np.uint16)
    for pixel in np.nditer(line_frame):
        histogram[(int)(pixel*histogram_steps/256)]+=1
    # send histogram as MIDI CC messages
    index=1
    for pixel in np.nditer(histogram[0:-conf['removed_top_bins']]):
        velocity=pixel-threshold
        if velocity>127:
            velocity=127
        if velocity<=0:
            velocity=0
        packet=((0xB<<4), index, velocity) # MIDI CC ("Continuous Controller") message
        midi_output_queue.append(packet)
        index+=1
    
    # shape output
    # gather depth data from a laser line projection
    gray=cv2.cvtColor(cropped_frame, cv2.COLOR_BGR2GRAY)
    grad_y = cv2.Sobel(gray, cv2.CV_8U, 0, 2, ksize=5, scale=-.1, delta=0, borderType=cv2.BORDER_DEFAULT)
    _, laser = cv2.threshold(grad_y, conf['laser_threshold'], 255, cv2.THRESH_BINARY)
    weights=np.arange(laser.shape[0],dtype=float)

    weighted=np.tensordot(laser,weights,[[0],[0]])
    count   =np.sum      (laser.astype(float),0)
    depths  =weighted / count
    
    depths=depths[count>0] # cut NaN's
    
    # cut floor (small percentile)
    baseline=np.percentile(depths,95)
    depths=depths[depths<baseline-2]
    
    depth_min   =np.percentile(depths,5)
    depth_mean  =np.mean(depths)
    depth_width =np.sum(depths>0)
    depth_height=baseline-depth_mean
    depth_roughness=np.std(depths)
    
    print("baseline "+str(baseline)+" min "+str(depth_min)+" mean "+str(depth_mean)+" width "+str(depth_width)+" height "+str(depth_height)+" roughness "+str(depth_roughness)) 

    # send out mean width and height of the scanned pieces by MIDI CC 64 and 65.
    midi_output_queue.append(((0xB<<4), 64, int(depth_height*conf['height_scale'])))
    midi_output_queue.append(((0xB<<4), 65, int(depth_width *conf['width_scale' ])))
    midi_output_queue.append(((0xB<<4), 66, int(depth_roughness *conf['roughness_scale' ])))

    # impaint max to laser image (preview)
    laser[int(baseline+1)][:]=100
    laser[int(depth_min)][:]=100

    cv2.imshow("Security Feed", cropped_frame)
    cv2.imshow("laser", laser)
    
    # if the `q` key is pressed, break from the lop
    key = cv2.waitKey(10) & 0xFF
    if key == ord("q"):
            break



